using System;
using System.Net;
using System.ServiceModel;
using System.Configuration;
using System.ServiceModel.Security;
using System.ServiceModel.Description;

using no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.adresse;
using no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.bygning;
using no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.kommune;
using no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.matrikkelenhet;
using no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store;
using no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.forretning;

namespace MatWSTester
{
    class Program
    {
        private static string endpointAdresse = null;
        private static String bruker = null;
        private static String pass = null;
        private static String TITLE = "Matrikkel WS eksempelklienten";
        private static String ERROR_MSG = "ERROR!";

        static void Main(string[] args) {
            Console.Title = TITLE;
            var appSettings = ConfigurationManager.AppSettings;
            if (appSettings.Count == 0 || appSettings["EndpointAddress"] == null || appSettings["EndpointAddress"].Length < 2) {
                Console.Write("Skriv inn endpoint addresse: ");
                endpointAdresse = Console.ReadLine();
            } else {
                endpointAdresse = appSettings["EndpointAddress"];
            }

            if (appSettings.Count == 0 || appSettings["UserName"] == null || appSettings["UserName"].Length < 2) {
                Console.Write("Skriv inn brukernavn: ");
                bruker = Console.ReadLine();
                pass = ReadPassword();
            } else {
                bruker = appSettings["UserName"];
                pass = appSettings["Password"];
            }

            Console.WriteLine("DENNE EKSEMPELKLIENTEN FINNER OBJEKTINFORMASJON OM EN MATRIKKELENHET");
            Console.Title = TITLE + ": RUNNING...";
            try {
                Console.ForegroundColor = ConsoleColor.Green;
                TestMatrikkelServiceWS();
            } catch (MessageSecurityException e) {
                writeMsg("Autorisasjon feilet. Sjekk at brukernavn eller passord er riktig. \nError msg:" + e.Message, ConsoleColor.Red, true);
                writeMsg(e.StackTrace, ConsoleColor.DarkRed, false);
            } catch (FaultException e) {
                writeMsg("Error msg: " + e.Message, ConsoleColor.Red, true);
                writeMsg(e.StackTrace, ConsoleColor.DarkRed, false);
            } catch (CommunicationException e) {
                writeMsg("Kunne ikke kontakte gitt URL! Sjekk at endpoint adresse er riktig. \nError msg:" + e.Message, ConsoleColor.Red, true);
                writeMsg(e.StackTrace, ConsoleColor.DarkRed, false);
            }

            if (!Console.Title.EndsWith(ERROR_MSG)) {
                Console.Title = TITLE + ": TASK COMPLETED";
            }
            Console.ResetColor();
            Console.WriteLine("\nPress enter for å avslutte!");
            Console.ReadKey();
        }

        private static void TestMatrikkelServiceWS()
        {
            BasicHttpBinding myBinding = GetBasicHttpBinding();
            MatrikkelContext matrikkelContext = GetMatrikkelContext();
            AdressesokModel adressesokModel = GetAdressesokModel();

            /*** Client: StoreServiceClient ***/
            StoreServiceClient storeService = new StoreServiceClient(myBinding, new EndpointAddress(endpointAdresse + "StoreServiceWS"));
            SetWSCredentials(storeService.ClientCredentials);

            /*** Client: KommuneServiceClient ***/
            KommuneServiceClient kommuneService = new KommuneServiceClient(myBinding, new EndpointAddress(endpointAdresse + "KommuneServiceWS"));
            SetWSCredentials(kommuneService.ClientCredentials);

            KommuneIdent kommuneident = new KommuneIdent();
            kommuneident.kommunenummer = adressesokModel.kommunenummer;

            KommuneId kommuneId = kommuneService.findKommuneIdForIdent(kommuneident, matrikkelContext);
            Kommune kommune = (Kommune) storeService.getObject(kommuneId, matrikkelContext);

            Console.WriteLine("Oppgitt vegadresse: " + adressesokModel.adressenavn + " " + adressesokModel.husnummer + adressesokModel.bokstav + " i kommunen " + kommune.kommunenavn + "\n");

            // Hent AdressesokModel fra GetAdressesokModel()

            /*** Client: AdresseServiceClient ***/
            AdresseServiceClient adresseService = new AdresseServiceClient(myBinding, new EndpointAddress(endpointAdresse + "AdresseServiceWS"));
            SetWSCredentials(adresseService.ClientCredentials);

            AdresseId[] adresseIds = adresseService.findAdresser(adressesokModel, matrikkelContext);
            AdresseId adresseId = adresseIds[0];

            /*** Client: MatrikkelenhetService ***/
            MatrikkelenhetServiceClient matrikkelenhetService = new MatrikkelenhetServiceClient(myBinding, new EndpointAddress(endpointAdresse + "MatrikkelenhetServiceWS"));
            SetWSCredentials(matrikkelenhetService.ClientCredentials);

            MatrikkelenhetId[] matrikkelenhetIds = matrikkelenhetService.findMatrikkelenheterForAdresse(adresseId, false, true, matrikkelContext);
            MatrikkelenhetId matrikkelenhetId = matrikkelenhetIds[0];

            Vegadresse vegadresse = (Vegadresse) storeService.getObject(adresseId, matrikkelContext);
            Matrikkelenhet matrikkelenhet = (Matrikkelenhet) storeService.getObject(matrikkelenhetId, matrikkelContext);

            Matrikkelnummer matrikkelnummer = matrikkelenhet.matrikkelnummer;
            String matrikkelnummerString = matrikkelnummer.gardsnummer + "/" + matrikkelnummer.bruksnummer + "/" + matrikkelnummer.festenummer + "/" + matrikkelnummer.seksjonsnummer;
            String bruksnavn = matrikkelenhet.bruksnavn;
            LocalDate etableringsDato = matrikkelenhet.etableringsdato;
            bool harKulturminne = matrikkelenhet.harKulturminne;
            String matrikkelenhetType = matrikkelenhet.GetType().Name;
            bool harAktiveFestegrunner = matrikkelenhet.harAktiveFestegrunner;
            bool harTinglyst = matrikkelenhet.tinglyst;

            TeigForMatrikkelenhet[] teigForMatrikkelenhetList = matrikkelenhet.teigerForMatrikkelenhet;
            int antTeiger = teigForMatrikkelenhetList.Length;
            Double areal = 0.0;
            foreach (TeigForMatrikkelenhet teigInList in teigForMatrikkelenhetList) {
                Teig teig = (Teig) storeService.getObject(teigInList.teigId, matrikkelContext);
                areal += teig.lagretBeregnetAreal;
            }

            Console.WriteLine("AdresseId: " + adresseId.value);
            Console.WriteLine("MatrikkelenhetId: " + matrikkelenhetId.value);
            Console.WriteLine("Matrikkelnummer: " + matrikkelnummerString);
            Console.WriteLine("Type: " + matrikkelenhetType);
            Console.WriteLine("Bruksnavn: " + bruksnavn);
            Console.WriteLine("Etableringsdato: " + etableringsDato.date);
            Console.WriteLine("Tinglyst?: " + harTinglyst);
            Console.WriteLine("Har kulturminne?: " + harKulturminne);
            Console.WriteLine("Aktive festegrunner?: " + harAktiveFestegrunner);
            Console.WriteLine("Antall teiger: " + antTeiger);
            Console.WriteLine("Areal: " + areal);

            /*** Client: BygningService ***/
            BygningServiceClient bygningService = new BygningServiceClient(myBinding, new EndpointAddress(endpointAdresse + "BygningServiceWS"));
            SetWSCredentials(bygningService.ClientCredentials);

            ByggId[] byggIdList = bygningService.findByggForMatrikkelenhet(matrikkelenhetId, matrikkelContext);
            MatrikkelBubbleObject[] matrikkelBubbleObjectListBygg = storeService.getObjects(byggIdList, matrikkelContext);
            Console.WriteLine("\nAntall bygg (bygningsendring eller en bygning) knyttet til matrikkelenhet: " + matrikkelBubbleObjectListBygg.Length + "\n");

            int i = 1;
            foreach (Bygg bygg in matrikkelBubbleObjectListBygg) {
                Console.WriteLine("Bygg " + (i++) + " type: " + bygg.GetType().Name);

                if (bygg is Bygning) {
                    BygningstypeKode bygningstype = (BygningstypeKode) storeService.getObject(((Bygning)bygg).bygningstypeKodeId, matrikkelContext);
                    Console.WriteLine("\tType: " + bygningstype.navn[0].value);
                } else if (bygg is Bygningsendring) {
                    BygningsendringsKode bygningsendringsKode = (BygningsendringsKode) storeService.getObject(((Bygningsendring)bygg).bygningsendringsKodeId, matrikkelContext);
                    Console.WriteLine("\tType: " + bygningsendringsKode.navn[0].value);
                }

                Console.WriteLine("\tStatus: ");
                foreach(BygningsstatusHistorikk bygningsstatusHistorikk in bygg.bygningsstatusHistorikker) {
                    BygningsstatusKode bygningsstatusKode = (BygningsstatusKode)storeService.getObject(bygningsstatusHistorikk.bygningsstatusKodeId, matrikkelContext);
                    Console.WriteLine("\t\t" + bygningsstatusKode.navn[0].value + " " + localDateToString(bygningsstatusHistorikk.dato));
                }

                Console.Write("\tBruksenheter til bygg: ");
                MatrikkelBubbleObject[] matrikkelBubbleObjectListBruksenhet = storeService.getObjects(bygg.bruksenhetIds, matrikkelContext);
                foreach (Bruksenhet bruksenhet in matrikkelBubbleObjectListBruksenhet) {
                    BruksenhetstypeKode bruksenhetstype = (BruksenhetstypeKode)storeService.getObject(bruksenhet.bruksenhetstypeKodeId, matrikkelContext);
                    Console.Write(bruksenhetstype.navn[0].value + " ");
                }
                Console.WriteLine("");
            }

            Console.WriteLine("\nEierforhold:\n");
            foreach (Eierforhold eierforhold in matrikkelenhet.eierforhold) {
                EierforholdKode eierforholdKode = (EierforholdKode)storeService.getObject(eierforhold.eierforholdKodeId, matrikkelContext);
                Console.Write(eierforhold.GetType().Name + ": " + eierforholdKode.navn[0].value);

                if (eierforhold is IkkeTinglystEierforhold) {
                    IkkeTinglystEierforhold ikkeTinglystEierforhold = (IkkeTinglystEierforhold) eierforhold;
                    //Henter eierandel
                    Console.Write("(" + ikkeTinglystEierforhold.andel.teller + "/" + ikkeTinglystEierforhold.andel.nevner + "), ");
                    Console.Write("dato fra: " + localDateToString(ikkeTinglystEierforhold.datoFra) + ", ");
                    if (ikkeTinglystEierforhold is PersonIkkeTinglystEierforhold) {
                        //Henter navn
                        PersonIkkeTinglystEierforhold personIkkeTinglyst = (PersonIkkeTinglystEierforhold) ikkeTinglystEierforhold;
                        Person person = (Person)storeService.getObject(personIkkeTinglyst.eierId, matrikkelContext);
                        Console.WriteLine(person.navn + "\n");
                    }
                } else if (eierforhold is TinglystEierforhold) {
                    TinglystEierforhold tinglystEierforhold = (TinglystEierforhold)eierforhold;
                    //Henter eierandel
                    Console.Write("(" + tinglystEierforhold.andel.teller + "/" + tinglystEierforhold.andel.nevner + "), ");
                    Console.Write("dato fra: " + localDateToString(tinglystEierforhold.datoFra) + ", ");
                    if(tinglystEierforhold is PersonTinglystEierforhold) {
                        //Henter navn
                        PersonTinglystEierforhold personTinglyst = (PersonTinglystEierforhold) tinglystEierforhold;
                        Person person = (Person)storeService.getObject(personTinglyst.eierId, matrikkelContext);
                        Console.WriteLine(person.navn);
                    }
                } else if (eierforhold is Kontaktinstans) {
                    //Henter kun navn
                    Kontaktinstans kontaktinstans = (Kontaktinstans)eierforhold;
                    Person person = (Person)storeService.getObject(kontaktinstans.eierId, matrikkelContext);
                    Console.WriteLine(", dato fra: " + localDateToString(kontaktinstans.datoFra) + " " + person.navn);
                }
            }

            /*** Client: ForretningService ***/
            ForretningServiceClient forretningService = new ForretningServiceClient(myBinding, new EndpointAddress(endpointAdresse + "ForretningServiceWS"));
            SetWSCredentials(forretningService.ClientCredentials);

            Console.WriteLine("\nForretninger:\n");
            ForretningId[] forretningIdList = forretningService.findForretningerForMatrikkelenhet(matrikkelenhetId, matrikkelContext);
            MatrikkelBubbleObject[] matrikkelBubbleObjectListForretning = storeService.getObjects(forretningIdList, matrikkelContext);

            foreach (MatrikkelenhetForretning forretning in matrikkelBubbleObjectListForretning) {
                ForretningstypeKode forretningsType = (ForretningstypeKode)storeService.getObject(forretning.forretningstypeKodeId, matrikkelContext);
                Console.WriteLine((++i) + ". " + forretningsType.navn[0].value + " " + localDateToString(forretning.forretningsdokumentdato));

                TinglysingsstatusKode tinglysingsstatus = (TinglysingsstatusKode)storeService.getObject(forretning.tinglysingsstatusKodeId, matrikkelContext);
                Console.WriteLine("\tTinglysingsstatus: " + tinglysingsstatus.navn[0].value);

                Console.WriteLine("\tInvolverte: ");
                foreach (ArealIForretning arealIForretning in forretning.arealIForretninger) {
                    RolleIForretningKode rolleIForretning = (RolleIForretningKode) storeService.getObject(arealIForretning.rolleId, matrikkelContext);
                    Matrikkelenhet matrikkelenhetIForretning = (Matrikkelenhet) storeService.getObject(arealIForretning.matrikkelenhetId, matrikkelContext);
                    Matrikkelnummer matrikkelnrIForretning = matrikkelenhetIForretning.matrikkelnummer;
                    Kommune kommuneIForretning = (Kommune) storeService.getObject(matrikkelnrIForretning.kommuneId, matrikkelContext);

                    String matrikkelnrString = kommuneIForretning.kommunenummer + " - " + matrikkelnrIForretning.gardsnummer + "/" + matrikkelnrIForretning.bruksnummer + "/" + matrikkelnrIForretning.festenummer + "/" + matrikkelnrIForretning.seksjonsnummer;
                    Console.WriteLine("\t\t" + matrikkelnrString + " (" + rolleIForretning.navn[0].value + ", " + arealIForretning.arealendring + ")");
                }
            }
        }

        private static void writeMsg(string msg, ConsoleColor color, bool isError) {
            ConsoleColor fColor = Console.ForegroundColor;
            Console.ForegroundColor = color;
            Console.WriteLine(msg);
            Console.ForegroundColor = fColor;
            if (isError) {
                Console.Title = TITLE + ": " + ERROR_MSG;
            }
        }

        private static String localDateToString(LocalDate localDate) {
            if (localDate == null && localDate.date == null) {
                return "";
            }
            if (localDate.date.ToString().Length <= 10) {
                return localDate.date.ToString();
            }
            return localDate.date.ToString().Substring(0, 10);
        }

        private static BasicHttpBinding GetBasicHttpBinding() {
            var appSettings = ConfigurationManager.AppSettings;
            long maxMessageSize = 2048000;
            if (appSettings.Count != 0 && appSettings["MaxMessageSize"] != null) {
                maxMessageSize = long.Parse(appSettings["MaxMessageSize"]);
            } else {
                writeMsg("Kunne ikke finne MaxMessageSize i appsetting. MaxMessageSize er satt til " + maxMessageSize, ConsoleColor.DarkYellow, false);
            }

            BasicHttpBinding myBinding = new BasicHttpBinding();
            myBinding.Security.Mode = BasicHttpSecurityMode.Transport;
            myBinding.Security.Transport.ClientCredentialType = HttpClientCredentialType.Basic;
            myBinding.MaxReceivedMessageSize = maxMessageSize;

            System.Net.ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;
            return myBinding;
        }

        private static void SetWSCredentials(ClientCredentials credentials) {
            credentials.UserName.UserName = bruker;
            credentials.UserName.Password = pass;
        }

        private static MatrikkelContext GetMatrikkelContext() {
            DateTime SNAPSHOT_VERSJON_DATO = new DateTime(9999, 1, 1, 0, 0, 0);

            MatrikkelContext matrikkelContext = new MatrikkelContext();
            matrikkelContext.locale = "no_NO";
            matrikkelContext.brukOriginaleKoordinater = true;
            KoordinatsystemKodeId koordinatsystemKodeId = new KoordinatsystemKodeId();
            koordinatsystemKodeId.value = 22;
            matrikkelContext.koordinatsystemKodeId = koordinatsystemKodeId;
            matrikkelContext.systemVersion = "trunk";
            matrikkelContext.klientIdentifikasjon = "Min testklient";
            Timestamp timestamp = new Timestamp();
            timestamp.timestamp = SNAPSHOT_VERSJON_DATO;
            matrikkelContext.snapshotVersion = timestamp;
            return matrikkelContext;
        }

        private static AdressesokModel GetAdressesokModel() {
            AdressesokModel adressesokModel = new AdressesokModel();
            adressesokModel.adressetype = AdressesokModelAdressetype.VEGADRESSE;
            adressesokModel.adressetypeSpecified = true;
            adressesokModel.adressenavn = "Høybyveien";
            adressesokModel.kommunenummer = "3007";
            adressesokModel.bokstav = "A";
            adressesokModel.husnummer = "11";
            return adressesokModel;
        }

        private static String ReadPassword() {
            Console.Write("Skriv inn passord: ");
            string password = "";
            do {
                ConsoleKeyInfo key = Console.ReadKey(true);
                if (key.Key != ConsoleKey.Backspace && key.Key != ConsoleKey.Enter) {
                    password += key.KeyChar;
                    Console.Write("*");
                } else {
                    if (key.Key == ConsoleKey.Backspace && password.Length > 0) {
                        password = password.Substring(0, (password.Length - 1));
                        Console.Write("\b \b");
                    } else if (key.Key == ConsoleKey.Enter) {
                        break;
                    }
                }
            } while (true);

            Console.WriteLine("");
            return password;
        }
    }
}
